﻿import {
  fetchArticleById,
  formatDate,
  getBodyHtml,
  getCategoryLabel,
  getImageUrlOrPlaceholder,
} from "@/lib/nilto";

type PageProps = {
  params: { id: string };
};

export default async function ArticlePage({ params }: PageProps) {
  const { id } = params;
  let article = null;
  let errorMessage = "";

  if (!id) {
    errorMessage = "記事IDがありません";
  } else {
    try {
      article = await fetchArticleById(id);
    } catch (error) {
      errorMessage = error instanceof Error ? error.message : "不明なエラーです";
    }
  }

  const bodyHtml = article ? getBodyHtml(article.body) : "";
  const category = article ? getCategoryLabel(article.category) : "";
  const publishedAt = article ? formatDate(article._published_at || article._created_at) : "";
  const imageUrl = article ? getImageUrlOrPlaceholder(article.image) : null;

  return (
    <>
      {errorMessage ? (
        <div className="error-box">{errorMessage}</div>
      ) : !article ? (
        <div className="error-box">記事が見つかりません。</div>
      ) : (
        <article>
          <div className="article-header">
            <div>
              <h1 className="article-title">{article.title || "無題の記事"}</h1>
              <div className="article-meta">
                {category && <span>{category}</span>}
                {article.author?.name && <span>{article.author.name}</span>}
                {publishedAt && <span>{publishedAt}</span>}
              </div>
            </div>
            <div className="article-hero">
              <img src={imageUrl} alt={article.title || "No image"} />
            </div>
            {article.author?.profile && (
              <div className="article-meta">
                <span>{article.author.profile}</span>
              </div>
            )}
          </div>
          <div
            className="article-body"
            dangerouslySetInnerHTML={{ __html: bodyHtml }}
          />
        </article>
      )}
      <a className="back-link" href="/">
        ← 記事一覧へ戻る
      </a>
    </>
  );
}
